define([
    'underscore',
    'backbone',
    'moment',
    'User',
    'modules/new-appointment-request/resources/direct/model',
    'modules/appointments/appointments-module',
    'modules/new-appointment-request/tests/helpers/general-helper',
], function(_, Backbone, moment, user, Model, module, helper) {
    'use strict';


    describe('Direct Scheduling Model', function() {
        var RESOURCE_NAME = 'ds-booked-appointments';
        var REGULAR_VISIT_TYPE = '9';
        var PATIENT_REQUEST = 'P';
        var DEFAULT_LEVEL = '1';
        var NONE = '';
        var DEFAULT_REQUEST_TYPE = 'NEXT_AVAILABLE_APPT';
        var DEFAULT_CARE_TYPE = 'REGULAR';
        var DEFAULT_APPT_KIND = 'TRADITIONAL';
        var DEFAULT_VIDEO_APPT_KIND = 'MOBILE_ANY';
        var TEST_EMAIL = 'test@email.com';
        var DEFAULT_VIDEO_VISIT_SECONDARY_STOP_CODE = '179';

        var DIRECT = {
            'apptLength': 20,
            'dateTime': '01/15/2032 08:00:00',
            'desiredDate': '02/16/2033 00:00:00',
            'purpose': 'To be healthy',
        };
        var NEW_TH_FACILITY = {
            name: 'Sacred Heart',
            institutionCode: '123AB',
            rootStationCode: '123',
        };
        var CLINIC_NAME = 'ER';
        var CLINIC_FRIENDLY_NAME = 'Friendly ER';
        var CLINIC_ID = '123';
        var model;

        NEW_TH_FACILITY.institutionTimezone = 'America/New_York';

        beforeEach(function() {
            model = new Model();
            helper.userSpy();
        });


        describe('Static Data', function() {
            it('has the correct resource name', function() {
                expect(model.resourceName).toBe(RESOURCE_NAME);
            });

            it('sets up the correct defaults', function() {
                expect(model.get('apptType')).toBe(PATIENT_REQUEST);
                expect(model.get('purpose')).toBe(REGULAR_VISIT_TYPE);
                expect(model.get('lvl')).toBe(DEFAULT_LEVEL);
                expect(model.get('ekg')).toBe(NONE);
                expect(model.get('xRay')).toBe(NONE);
                expect(model.get('lab')).toBe(NONE);
            });
        });


        describe('getUrl', function() {
            beforeEach(function() {
                model.set('facility', new Backbone.Model({
                    institutionCode: '123AB',
                    rootStationCode: '123',
                }));
            });

            it('generates the correct url', function() {
                var expectedUrl = '../VeteranAppointmentRequestService/v4/rest/direct-scheduling/site/123/patient/ICN/123/booked-appointments';
                expect(model.getUrl()).toBe(expectedUrl);
            });
        });

        // Telehealth features unit test changes
        describe('createSaveData', function() {
            beforeEach(function() {
                var clinic;

                model.set('direct', DIRECT);
                model.set('facility', new Backbone.Model(NEW_TH_FACILITY));
                clinic = new Backbone.Model({
                    clinicName: CLINIC_NAME,
                    clinicId: CLINIC_ID,
                });
                clinic.getClinicName = function() {
                    return clinic.get('clinicName');
                };
                model.set('clinic', clinic);
                model.set('emailPreferences', new Backbone.Model({emailAddress: TEST_EMAIL}));
            });

            it('creates the correct direct data', function() {
                var direct = model._directSaveData();

                expect(direct.duration).toBe(DIRECT.apptLength);
                expect(direct.dateTime).toBe(moment.utc(DIRECT.dateTime, 'MM/DD/YYYY HH:mm:ss').format());
                expect(direct.desiredDate).toBe(moment.utc(DIRECT.desiredDate, 'MM/DD/YYYY HH:mm:ss').format());
                expect(direct.bookingNotes).toBe(DIRECT.purpose);
            });

            it('creates the correct patient data', function() {
                var patientData = model._patientSaveData().patients.patient[0];
                var patientId = patientData.id;
                var patientName = patientData.name;
                var patientContactInformation = patientData.contactInformation;
                var patientLocation = patientData.location;

                expect(patientId.uniqueId).toBe(user.get('id'));
                expect(patientId.assigningAuthority).toBe(user.get('idType'));

                expect(patientName.firstName).toBe(user.get('firstName'));
                expect(patientName.lastName).toBe(user.get('lastName'));
                expect(patientName.middleInitial).toBe(user.get('middleInitial'));

                expect(patientContactInformation.preferredEmail).toBe(TEST_EMAIL);
                expect(patientContactInformation.timeZone).toBe(NEW_TH_FACILITY.institutionTimezone);

                expect(patientLocation.type).toBe('VA');
                expect(patientLocation.facility.name).toBe(model.getFacilityName());
                expect(patientLocation.facility.siteCode).toBe(model.getSiteCode());
                expect(patientLocation.facility.timeZone).toBe(model.getInstitutionTimezone());

                expect(patientLocation.clinic.ien).toBe(model.getClinicId());
                expect(patientLocation.clinic.name).toBe(model.getClinicName());
            });

            it('creates the correct appointment type/kind data', function() {
                var appointmentTypeData = model._appointmentTypeSaveData();

                expect(appointmentTypeData.schedulingRequestType).toBe(DEFAULT_REQUEST_TYPE);
                expect(appointmentTypeData.type).toBe(DEFAULT_CARE_TYPE);
                expect(appointmentTypeData.appointmentKind).toBe(DEFAULT_APPT_KIND);
            });

            it('creates the correct provider data', function() {
                var providerData = model._providersSaveData().providers.provider[0];

                expect(providerData.location.type).toBe('VA');
            });

            it('puts all the data together correctly', function() {
                var postData = model.createSaveData();
                var patientPostData = postData.patients.patient[0];

                expect(postData.dateTime).toBe(moment.utc(DIRECT.dateTime, 'MM/DD/YYYY HH:mm').format());
                expect(postData.desiredDate).toBe(moment.utc(DIRECT.desiredDate, 'MM/DD/YYYY HH:mm').format());
                expect(postData.bookingNotes).toBe(DIRECT.purpose);
                expect(postData.duration).toBe(DIRECT.apptLength);

                expect(patientPostData.id.assigningAuthority).toBe(user.get('idType'));
                expect(patientPostData.id.uniqueId).toBe(user.get('id'));

                expect(patientPostData.name.firstName).toBe(user.get('firstName'));
                expect(patientPostData.name.lastName).toBe(user.get('lastName'));
                expect(patientPostData.name.middleInitial).toBe(user.get('middleInitial'));

                expect(patientPostData.contactInformation.preferredEmail).toBe(TEST_EMAIL);
                expect(patientPostData.contactInformation.timeZone).toBe(NEW_TH_FACILITY.institutionTimezone);

                expect(patientPostData.location.type).toBe('VA');
                expect(patientPostData.location.facility.name).toBe(model.getFacilityName());
                expect(patientPostData.location.facility.siteCode).toBe(model.getSiteCode());
                expect(patientPostData.location.facility.timeZone).toBe(model.getInstitutionTimezone());

                expect(patientPostData.location.clinic.ien).toBe(model.getClinicId());
                expect(patientPostData.location.clinic.name).toBe(model.getClinicName());

                expect(postData.schedulingRequestType).toBe(DEFAULT_REQUEST_TYPE);
                expect(postData.type).toBe(DEFAULT_CARE_TYPE);
                expect(postData.appointmentKind).toBe(DEFAULT_APPT_KIND);

                expect(postData.apptType).toBe(PATIENT_REQUEST);
                expect(postData.purpose).toBe(REGULAR_VISIT_TYPE);
                expect(postData.lvl).toBe(DEFAULT_LEVEL);
                expect(postData.ekg).toBe(NONE);
                expect(postData.xRay).toBe(NONE);
                expect(postData.lab).toBe(NONE);
            });
        });

        describe('createSaveData for video visit', function() {
            beforeEach(function() {
                var clinic;

                model.set('direct', DIRECT);
                model.set('facility', new Backbone.Model(NEW_TH_FACILITY));
                clinic = new Backbone.Model({
                    clinicName: CLINIC_NAME,
                    clinicId: CLINIC_ID,
                    secondaryStopCode: DEFAULT_VIDEO_VISIT_SECONDARY_STOP_CODE,
                });
                clinic.getClinicName = function() {
                    return clinic.get('clinicName');
                };
                model.set('clinic', clinic);
                model.set('emailPreferences', new Backbone.Model({emailAddress: TEST_EMAIL}));
            });

            it('creates the correct direct data', function() {
                var direct = model._directSaveData();

                expect(direct.duration).toBe(DIRECT.apptLength);
                expect(direct.dateTime).toBe(moment(DIRECT.dateTime).format());
                expect(direct.desiredDate).toBe(moment(DIRECT.desiredDate).format());
                expect(direct.bookingNotes).toBe(DIRECT.purpose);
            });

            it('creates the correct patient data', function() {
                var patientData = model._patientSaveData().patients.patient[0];
                var patientId = patientData.id;
                var patientName = patientData.name;
                var patientContactInformation = patientData.contactInformation;
                var patientLocation = patientData.location;

                expect(patientId.uniqueId).toBe(user.get('id'));
                expect(patientId.assigningAuthority).toBe(user.get('idType'));

                expect(patientName.firstName).toBe(user.get('firstName'));
                expect(patientName.lastName).toBe(user.get('lastName'));
                expect(patientName.middleInitial).toBe(user.get('middleInitial'));

                expect(patientContactInformation.preferredEmail).toBe(TEST_EMAIL);
                expect(patientContactInformation.timeZone).toBe(NEW_TH_FACILITY.institutionTimezone);

                expect(patientLocation.type).toBe('NonVA');
                expect(patientLocation.facility.name).toBe(model.getFacilityName());
                expect(patientLocation.facility.siteCode).toBe(model.getSiteCode());
                expect(patientLocation.facility.timeZone).toBe(model.getInstitutionTimezone());

                expect(patientLocation.clinic.ien).toBe(model.getClinicId());
                expect(patientLocation.clinic.name).toBe(model.getClinicName());
            });

            it('creates the correct appointment type/kind data', function() {
                var appointmentTypeData = model._appointmentTypeSaveData();

                expect(appointmentTypeData.schedulingRequestType).toBe(DEFAULT_REQUEST_TYPE);
                expect(appointmentTypeData.type).toBe(DEFAULT_CARE_TYPE);
                expect(appointmentTypeData.appointmentKind).toBe(DEFAULT_VIDEO_APPT_KIND);
            });

            it('creates the correct provider data', function() {
                var providerData = model._providersSaveData().providers.provider[0];

                expect(providerData.location.type).toBe('VA');
            });

            it('puts all the data together correctly', function() {
                var postData = model.createSaveData();
                var patientPostData = postData.patients.patient[0];

                expect(postData.dateTime).toBe(moment(DIRECT.dateTime).format());
                expect(postData.desiredDate).toBe(moment(DIRECT.desiredDate).format());
                expect(postData.bookingNotes).toBe(DIRECT.purpose);
                expect(postData.duration).toBe(DIRECT.apptLength);

                expect(patientPostData.id.assigningAuthority).toBe(user.get('idType'));
                expect(patientPostData.id.uniqueId).toBe(user.get('id'));

                expect(patientPostData.name.firstName).toBe(user.get('firstName'));
                expect(patientPostData.name.lastName).toBe(user.get('lastName'));
                expect(patientPostData.name.middleInitial).toBe(user.get('middleInitial'));

                expect(patientPostData.contactInformation.preferredEmail).toBe(TEST_EMAIL);
                expect(patientPostData.contactInformation.timeZone).toBe(NEW_TH_FACILITY.institutionTimezone);

                expect(patientPostData.location.type).toBe('NonVA');
                expect(patientPostData.location.facility.name).toBe(model.getFacilityName());
                expect(patientPostData.location.facility.siteCode).toBe(model.getSiteCode());
                expect(patientPostData.location.facility.timeZone).toBe(model.getInstitutionTimezone());

                expect(patientPostData.location.clinic.ien).toBe(model.getClinicId());
                expect(patientPostData.location.clinic.name).toBe(model.getClinicName());

                expect(postData.schedulingRequestType).toBe(DEFAULT_REQUEST_TYPE);
                expect(postData.type).toBe(DEFAULT_CARE_TYPE);
                expect(postData.appointmentKind).toBe(DEFAULT_VIDEO_APPT_KIND);

                expect(postData.apptType).toBe(PATIENT_REQUEST);
                expect(postData.purpose).toBe(REGULAR_VISIT_TYPE);
                expect(postData.lvl).toBe(DEFAULT_LEVEL);
                expect(postData.ekg).toBe(NONE);
                expect(postData.xRay).toBe(NONE);
                expect(postData.lab).toBe(NONE);
            });
        });
        describe('save', function() {
            var spy;

            beforeEach(function() {
                var clinic;

                spy = spyOn(Backbone.Model.prototype, 'save').and.callFake(_.noop);
                model.set('direct', DIRECT);
                model.set('facility', new Backbone.Model(NEW_TH_FACILITY));
                clinic = new Backbone.Model({
                    clinicName: CLINIC_NAME,
                    clinicId: CLINIC_ID,
                });
                clinic.getClinicName = function() {
                    return clinic.get('clinicName');
                };
                model.set('clinic', clinic);
                model.set('emailPreferences', new Backbone.Model({emailAddress: TEST_EMAIL}));
            });

            it('saves with the correct data', function() {
                var attributes;
                var options;

                model.save();
                attributes = {
                    duration: 20,
                    dateTime: moment.utc(DIRECT.dateTime, 'MM/DD/YYYY HH:mm').format(),
                    desiredDate: moment.utc(DIRECT.desiredDate, 'MM/DD/YYYY HH:mm').format(),
                    bookingNotes: 'To be healthy',
                    apptType: 'P',
                    purpose: '9',
                    ekg: '',
                    lab: '',
                    lvl: '1',
                    xRay: '',
                    patients: {
                        patient: [
                            {
                                id: {
                                    uniqueId: '123',
                                    assigningAuthority: 'ICN',
                                },
                                name: {
                                    firstName: user.get('firstName'),
                                    lastName: user.get('lastName'),
                                    middleInitial: user.get('middleInitial'),
                                },
                                contactInformation: {
                                    preferredEmail: TEST_EMAIL,
                                    timeZone: NEW_TH_FACILITY.institutionTimezone,
                                },
                                location: {
                                    type: 'VA',
                                    facility: {
                                        name: model.getFacilityName(),
                                        siteCode: model.getSiteCode(),
                                        timeZone: NEW_TH_FACILITY.institutionTimezone,
                                    },
                                    clinic: {
                                        ien: model.getClinicId(),
                                        name: model.getClinicName(),
                                    },
                                },
                            },
                        ],
                    },
                    schedulingRequestType: DEFAULT_REQUEST_TYPE,
                    type: DEFAULT_CARE_TYPE,
                    appointmentKind: DEFAULT_APPT_KIND,
                    providers: {provider: [{location: {type: 'VA'}}]},
                };
                options = {
                    contentType: 'application/json',
                    dataType: 'json',
                    traditional: true,
                    wait: true,
                    url: '../VeteranAppointmentRequestService/v4/rest/direct-scheduling/site/123/patient/ICN/123/booked-appointments',
                };
                expect(spy).toHaveBeenCalledWith(attributes, options);
            });
        }
        );

        describe('toAppointmentsModel', function() {

            var Clinic = Backbone.Model.extend({
                getClinicName: function() {
                    return this.get('clinicName');
                },
                getFriendlyLocationName: function() {
                    return this.get('clinicFriendlyLocationName');
                },
            });


            beforeEach(function() {

                model = new Model({
                    direct: {
                        apptLength: '1h',
                        dateTime: '01/02/2000',
                    },
                    clinic: new Clinic({
                        clinicId: '123',
                        clinicName: 'Sacred Heart',
                        clinicFriendlyLocationName: CLINIC_FRIENDLY_NAME,
                    }),
                    facility: new Backbone.Model({
                        institutionCode: '123AB',
                        rootStationCode: '123',
                        name: 'Project MKUltra',
                    }),
                });
            });

            it('casts to appointment model', function() {
                var am = model.toAppointmentsModel();
                expect(am instanceof module.appointments.model).toBeTruthy();
            });

            it('fills the model will all required data', function() {
                var am = model.toAppointmentsModel();
                expect(am.get('link')).toEqual({
                    href: '../VeteranAppointmentRequestService/v4/rest/direct-scheduling/site/123/patient/ICN/123/cancel-appointment',
                    rel: 'related',
                    title: 'cancel-appointment',
                });
                expect(am.get('currentStatus')).toBe('Scheduled');
                expect(am.get('siteCode')).toBe('123');
                expect(am.get('facilityName')).toBe('Project MKUltra');
                expect(am.get('clinic')).toEqual({
                    id: '123',
                    friendlyName: 'Sacred Heart',
                    clinicFriendlyName: CLINIC_FRIENDLY_NAME,
                });
                expect(am.get('appointmentLength')).toBe('1h');
                expect(am.get('appointmentTime')).toBe('01/02/2000');

            });
        });
    });
});
